/* --------------------------------------------------------------
  UploadStyleModal.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


/* globals -Modal */

'use strict';

import Modal from '../libs/Modal';
import StyleUpload from '../widgets/StyleUpload';
import EventsEmitter from '../libs/EventsEmitter';

/**
 * Upload Style Modal Controller
 *
 * This controller will handle the style upload operation. The upload will be done
 * with AJAX and thus the use of the StyleUpload widget is required.
 */
export default class UploadStyleModal {
	/**
	 * Class Constructor
	 */
	constructor() {
		/**
		 * The modal selector.
		 *
		 * @type {jQuery}
		 */
		this.$modal = null;
		
		/**
		 * The StyleUpload Instance
		 *
		 * @type {StyleUpload}
		 */
		this.styleUpload = null;
	}
	
	/**
	 * Initialize Controller
	 */
	initialize() {
		const data = {
			title_upload_style_modal: StyleEdit.Language.translate('title_upload_style_modal', 'style_edit'),
			placeholder_style_name: StyleEdit.Language.translate('placeholder_style_name', 'style_edit'),
			placeholder_browse: StyleEdit.Language.translate('placeholder_browse', 'style_edit'),
			option_create: StyleEdit.Language.translate('option_create', 'style_edit'),
			option_cancel: StyleEdit.Language.translate('option_cancel', 'style_edit')
		};
		
		this.$modal = Modal.show($('#upload-style-modal-template'), data);
		
		// Focus style-name input field.
		this.$modal.find('input[name="style-name"]').focus();
		
		// Initialize the file upload.
		this.styleUpload = new StyleUpload(this.$modal.find('#json-upload'),
			StyleEdit.Config.get('baseUrl') + '/api.php/templates/' + StyleEdit.Config.get('template') + '/upload'
			+ (StyleEdit.Config.get('theme') ? '?theme' : ''));
		
		this._bindEventHandlers();
		
		EventsEmitter.triggerControllerInitialized(this.$modal, ['UploadStyleModal']);
		
		$.material.init();
	}
	
	/**
	 * Destroy Controller
	 */
	destroy() {
		Modal.hide(this.$modal);
		EventsEmitter.triggerControllerDestroyed(this.$modal, ['UploadStyleModal']);
	}
	
	/**
	 * Event: Click Browse Button
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onClickBrowse(event) {
		const fileName = $(event.currentTarget).val();
		$(event.currentTarget).siblings('input:text').attr('placeholder', fileName);
	}
	
	/**
	 * Event: Create Button Click
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onClickCreate(event) {
		const $element = $(event.currentTarget);
		const $newStyleName = this.$modal.find('#style-name');
		const $newStyleFile = this.$modal.find('#json-upload');
		
		$newStyleName.parent().removeClass('has-error');
		$newStyleFile.parent().removeClass('has-error is-focused');
		
		if ($newStyleName.val() === '') {
			$newStyleName.parent().addClass('has-error');
			$newStyleName.focus();
			return;
		}
		
		if ($newStyleFile.val() === '') {
			$newStyleFile.parent().addClass('has-error is-focused');
			$newStyleFile.focus();
			return;
		}
		
		this.styleUpload.upload($newStyleName.val()).done(response => {
			EventsEmitter.triggerStyleCreated($element, [response]);
			Modal.hide(this.$modal); // close the modal layer
		});
	}
	
	/**
	 * Event: Cancel Button Click
	 *
	 * @private
	 */
	_onClickCancel() {
		this.destroy();
	}
	
	/**
	 * Bind the require event handlers.
	 *
	 * @private
	 */
	_bindEventHandlers() {
		this.$modal
			.on('click', '.btn.create', event => this._onClickCreate(event))
			.on('click', '.btn.cancel', () => this._onClickCancel())
			.on('change', '#json-upload', event => this._onClickBrowse(event));
	}
}
